package gov.va.med.mhv.journal.service.impl;

import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.journal.converter.FoodJournalConverter;
import gov.va.med.mhv.journal.converter.MealItemConverter;
import gov.va.med.mhv.journal.data.model.FoodJournal;
import gov.va.med.mhv.journal.data.model.MealItem;
import gov.va.med.mhv.journal.dto.FoodJournalDTO;
import gov.va.med.mhv.journal.dto.MealItemDTO;
import gov.va.med.mhv.journal.repository.FoodJournalRepository;
import gov.va.med.mhv.journal.repository.MealItemRepository;
import gov.va.med.mhv.journal.service.FoodJournalService;
import gov.va.med.mhv.journal.service.validator.FoodJournalValidator;
import gov.va.med.mhv.journal.service.validator.MealItemValidator;

@Component
public class FoodJournalServiceImpl implements FoodJournalService {
	private static Logger log = LogManager.getLogger(FoodJournalServiceImpl.class);

	@Autowired
	private FoodJournalRepository foodJournalRepository;

	@Autowired
	private MealItemRepository mealItemRepository;

	@Autowired
	private FoodJournalConverter foodJournalConverter;

	@Autowired
	private MealItemConverter mealItemConverter;

	@Autowired
	private FoodJournalValidator foodJournalValidator;

	@Autowired
	private MealItemValidator mealItemValidator;

	@Override
	@Transactional
	public List<FoodJournalDTO> findFoodJournalsForUser(Long userProfileId) throws MHVException {
		List<FoodJournal> list = null;
		List<FoodJournalDTO> dtoList = null;

		try {
			list = foodJournalRepository.findFoodJournalsForUser(userProfileId);

			if (list != null) {
				dtoList = new ArrayList<FoodJournalDTO>();
				for (FoodJournal foodJournal : list) {
					dtoList.add(foodJournalConverter.convertDomainFoodJournal(foodJournal));
				}
			}
		} catch (Exception e) {
			log.error(e);
			throw new MHVException("Error in fetching foodJournals " + e.getMessage(), e);
		}

		return dtoList;
	}

	@Override
	@Transactional
	public FoodJournalDTO findFoodJournalById(Long foodJournalId) throws MHVException {
		FoodJournal foodJournal = null;
		FoodJournalDTO foodJournalDTO = null;

		try {
			foodJournal = foodJournalRepository.findOne(foodJournalId);

			if (foodJournal != null) {
				foodJournalDTO = foodJournalConverter.convertDomainFoodJournal(foodJournal);
			}
		} catch (Exception e) {
			log.error(e);
			throw new MHVException("Error in fetching foodJournal " + e.getMessage(), e);
		}

		return foodJournalDTO;
	}

	@Override
	@Transactional
	public MealItemDTO getMealItemById(Long mealItemId) throws MHVException {
		MealItem mealItem = null;
		MealItemDTO mealItemDTO = null;

		try {
			mealItem = mealItemRepository.findOne(mealItemId);
			if (null != mealItem) {
				mealItemDTO = mealItemConverter.convertDomainMealItem(mealItem);
			}

		} catch (Exception e) {
			log.error(e);
			throw new MHVException("Error in fetching mealItem " + e.getMessage(), e);
		}

		return mealItemDTO;
	}

	@Override
	@Transactional
	public List<FoodJournalDTO> getFoodJournalsForDashBoard(Long userProfileId) throws MHVException {
		List<FoodJournal> list = null;
		List<FoodJournalDTO> dtoList = null;

		try {
			Pageable topFive = new PageRequest(0, 5);
			list = foodJournalRepository.getDashBoardFoodJournalsForUser(userProfileId, topFive);
			if (list != null) {
				dtoList = new ArrayList<FoodJournalDTO>();
				for (FoodJournal foodJournal : list) {
					dtoList.add(foodJournalConverter.convertDomainFoodJournal(foodJournal));
				}
			}
		} catch (Exception e) {
			log.error(e);
			throw new MHVException("Error in fetching top5 foodJournals " + e.getMessage(), e);
		}

		return dtoList;
	}

	@Override
	@Transactional
	public MealItemDTO addMealItem(MealItemDTO mealItemDTO) throws MHVException {
		ResponseUtil<MealItemDTO> response = new ResponseUtil<MealItemDTO>();
		MealItem mealItem = null;
		FoodJournal foodJournal = null;
		MealItemDTO dto = null;

		try {
			response = mealItemValidator.fieldValidation(mealItemDTO);

			// input validation errors
			if (response.getValidationErrors().size() > 0) {
				throw new MHVException(response);
			}

			mealItem = mealItemConverter.convertDTOMealItem(mealItemDTO);
			foodJournal = foodJournalRepository.findOne(mealItem.getFoodJournalId());
			mealItem.setFoodJournal(foodJournal);
			mealItem = mealItemRepository.save(mealItem);

			dto = mealItemConverter.convertDomainMealItem(mealItem);

		} catch (Exception e) {
			log.error(e);
			throw new MHVException("Error in storing meal Item in DB " + e.getMessage(), e);
		}

		return dto;
	}

	@Override
	@Transactional
	public FoodJournalDTO addFoodJournal(FoodJournalDTO foodJournalDTO) throws MHVException {
		ResponseUtil<FoodJournalDTO> response = new ResponseUtil<FoodJournalDTO>();
		FoodJournal foodJournal = null;
		FoodJournalDTO dto = null;

		response = foodJournalValidator.foodJournalValidations(foodJournalDTO);

		// input validation errors
		if (response.getValidationErrors().size() > 0) {
			throw new MHVException(response);
		}

		try {
			foodJournal = foodJournalConverter.convertDTOFoodJournal(foodJournalDTO);

			if (foodJournal.getMealItems() != null) {
				for (MealItem mealItem : foodJournal.getMealItems()) {
					mealItem.setFoodJournal(foodJournal);
				}
			}

			// update only Food Journal info
			/*
			 * if (null != foodJournal.getFoodJournalId() && null !=
			 * foodJournal.getMealItems()) { foodJournal.getMealItems().clear();
			 * }
			 */

			foodJournal = foodJournalRepository.save(foodJournal);

			// remove foodJournal object from mealItems to avoid circular
			// references
			if (foodJournal.getMealItems() != null) {
				Long foodJounralId = (null != foodJournal.getFoodJournalId()) ? foodJournal.getFoodJournalId() : null;
				for (MealItem mealItem : foodJournal.getMealItems()) {
					mealItem.setFoodJournalId(foodJounralId);
				}
			}
			dto = foodJournalConverter.convertDomainFoodJournal(foodJournal);

		} catch (Exception e) {
			log.error(e);
			throw new MHVException("Error in storing meal Item in DB " + e.getMessage(), e);
		}

		return dto;
	}

	@Override
	@Transactional
	public void deleteFoodJournal(Long foodJournalId) throws MHVException {

		try {
			foodJournalRepository.delete(foodJournalId);
		} catch (Exception e) {
			log.error(e);
			throw new MHVException("Error in deleting foodJournal " + e.getMessage(), e);
		}
	}

	@Override
	@Transactional
	public void deleteMealItem(Long mealItemId) throws MHVException {
		MealItem mealItem = null;
		FoodJournal foodJournal = null;
		Long foodJournalId = null;

		try {
			mealItem = mealItemRepository.findOne(mealItemId);
			foodJournalId = mealItem.getFoodJournalId();
			foodJournal = foodJournalRepository.findOne(foodJournalId);
			foodJournal.getMealItems().remove(mealItem);
			foodJournalRepository.save(foodJournal);
		} catch (Exception e) {
			log.error(e);
			throw new MHVException("Error in deleting meal Item " + e.getMessage(), e);
		}
	}

}
